# Codebase Scan Summary

## 🎯 Project Overview
**Framework:** Symfony 7.3 + React 19.2 with TypeScript  
**Architecture:** Symfony backend with React frontend using Stimulus controllers  
**Styling:** TailwindCSS v4 + shadcn/ui (New York variant)  
**State Management:** React Query (@tanstack/react-query)  
**Routing:** React Router DOM with Symfony integration

---

## 📁 Project Structure

### Frontend (React) Structure
```
assets/
├── controllers/
│   ├── front_controller.js     # Stimulus controller for frontend
│   ├── user_controller.js      # Stimulus controller for user admin
│   └── react/
│       ├── components/
│       │   └── ui/             # shadcn components
│       │       └── button.tsx  # Example shadcn button component
│       ├── front/
│       │   └── App.js          # Main frontend React app
│       ├── user/
│       │   └── App.js          # User admin React app
│       └── lib/
│           └── utils.ts        # Utility functions (cn helper)
├── styles/
│   └── app.css                 # TailwindCSS v4 configuration
└── app.js                      # Main entry point
```

### Backend (Symfony) Structure
```
src/
├── Controller/
│   ├── FrontController.php     # Frontend route handler (/{react})
│   ├── UserController.php      # User admin route handler (/user/{react})
│   ├── AdminController.php     # Admin route handler (/admin/{react})
│   └── Admin/
│       └── SettingsController.php
├── Entity/
│   ├── User.php                # User entity with SMTP fields
│   ├── Settings.php
│   └── ResetPasswordToken.php
└── Service/
    ├── EmailVerifier.php
    ├── ResetPasswordHelper.php
    └── DynamicMailer.php
```

### Templates (Twig)
```
templates/
├── base.html.twig              # Base layout with webpack assets
├── front/
│   └── index.html.twig         # Frontend React mount point
├── user/
│   └── index.html.twig         # User admin React mount point
└── security/                   # Auth pages (login, register, etc.)
```

---

## 🔧 Technology Stack

### Dependencies Installed
**React & UI:**
- `react@19.2.0` & `react-dom@19.2.0`
- `@radix-ui/react-slot@1.2.3` (for shadcn)
- `lucide-react@0.545.0` (icons)
- `react-router-dom@7.9.4`

**Styling:**
- `tailwindcss@4.1.14`
- `@tailwindcss/postcss@4.1.14`
- `tailwind-merge@3.3.1` (for cn utility)
- `class-variance-authority@0.7.1` (CVA for component variants)
- `clsx@2.1.1`
- `tw-animate-css@1.4.0`

**State Management:**
- `@tanstack/react-query@5.90.2`

**Build Tools:**
- `@symfony/webpack-encore@5.0.0`
- `webpack@5.74.0`
- `@babel/preset-react@7.27.1`
- `ts-loader@9.5.4`
- `postcss@8.5.6` & `postcss-loader@8.2.0`

**Symfony:**
- Symfony 7.3.* framework
- Doctrine ORM 3.5
- Stimulus Bridge

---

## ⚙️ Configuration Files

### TailwindCSS v4 Setup (`assets/styles/app.css`)
```css
@import "tailwindcss";
@import "tw-animate-css";

@custom-variant dark (&:is(.dark *));
@source not "../../public";

@theme inline {
  /* CSS variables for shadcn theming */
  --color-background: var(--background);
  --color-foreground: var(--foreground);
  /* ...other theme variables */
}
```

### shadcn Configuration (`components.json`)
- **Style:** New York
- **Base Color:** neutral
- **CSS Variables:** Enabled
- **Icon Library:** lucide
- **Aliases:**
  - `@/components` → components
  - `@/utils` → lib/utils
  - `@/lib` → lib

### Webpack Config (`webpack.config.js`)
- **Alias:** `@` → `assets/controllers/react`
- **Output:** `public/build/`
- **PostCSS:** Enabled for TailwindCSS v4
- **TypeScript:** Enabled
- **React Preset:** Enabled
- **Stimulus Bridge:** Enabled

### TypeScript Config (`tsconfig.json`)
- **JSX:** react-jsx
- **Module:** ESNext
- **Path Mapping:** `@/*` → `assets/controllers/react/*`

---

## 🚦 Routing Structure

### Frontend Routes
**Controller:** `FrontController.php`  
**Route Pattern:** `/{react}` (catches all routes except protected ones)  
**Twig Template:** `templates/front/index.html.twig`  
**React App:** `assets/controllers/react/front/App.js`

### User Admin Routes
**Controller:** `UserController.php`  
**Route Pattern:** `/user/{react}`  
**Twig Template:** `templates/user/index.html.twig`  
**React App:** `assets/controllers/react/user/App.js`

### Admin Routes
**Controller:** `AdminController.php`  
**Route Pattern:** `/admin/{react}`

---

## 🔑 Key Features

### Authentication System
- User registration with email verification
- Login/logout with "remember me"
- Password reset via email
- CSRF protection
- Secure password hashing

### User Entity Fields
```php
- id (int)
- email (string, unique)
- roles (array)
- password (string, hashed)
- isVerified (bool)
- createdAt (DateTimeImmutable)
- smtpHost, smtpPort, smtpEncryption (nullable)
- smtpUsername, smtpPassword (nullable)
- smtpFromEmail, smtpFromName (nullable)
```

---

## 🎨 Current shadcn Components
- ✅ **Button** (`@/components/ui/button.tsx`)
  - Variants: default, destructive, outline, secondary, ghost, link
  - Sizes: default, sm, lg, icon, icon-sm, icon-lg

---

## 📦 Build Process

### Development
```bash
npm run watch          # Watch mode for development
npm run dev            # Build for development
npm run dev-server     # Start Encore dev server
```

### Production
```bash
npm run build          # Build for production
```

---

## 🧩 React Integration Pattern

### How Symfony + React Works
1. **Symfony** serves a Twig template
2. **Stimulus controller** mounts the React app
3. **React Router** handles client-side routing
4. **React Query** manages API state

### Example Integration
```twig
{# templates/front/index.html.twig #}
<div {{stimulus_controller('front', {
    siteUrl: path('app_front')
})}}></div>
```

```javascript
// assets/controllers/front_controller.js
export default class extends Controller {
    connect() {
        const root = createRoot(this.element);
        const queryClient = new QueryClient();
        
        root.render(
            <QueryClientProvider client={queryClient}>
                <Router basename={this.siteUrlValue}>
                    <App />
                </Router>
            </QueryClientProvider>
        );
    }
}
```

---

## 📝 Ready for Implementation

### Pages to Build
1. ✅ **Home** - Frontend
2. ✅ **About** - Frontend
3. ✅ **CV** - Frontend
4. ✅ **Showreel** - Frontend
5. ✅ **Voice Reel** - Frontend
6. ✅ **Gallery** - Frontend
7. ✅ **Contact** - Frontend

### Location for React Components
```
assets/controllers/react/front/
├── App.js                    # Main app with routing
├── pages/
│   ├── Home.jsx             # Home page
│   ├── About.jsx            # About page
│   ├── CV.jsx               # CV page
│   ├── Showreel.jsx         # Showreel page
│   ├── VoiceReel.jsx        # Voice reel page
│   ├── Gallery.jsx          # Gallery page
│   └── Contact.jsx          # Contact page
├── components/
│   ├── layout/
│   │   ├── Header.jsx       # Navigation header
│   │   ├── Footer.jsx       # Footer
│   │   └── Layout.jsx       # Layout wrapper
│   └── shared/              # Shared components
└── hooks/                   # Custom hooks if needed
```

---

## ✅ Summary

**System is ready for:**
- ✅ TailwindCSS v4 styling
- ✅ shadcn/ui component integration
- ✅ React 19.2 with TypeScript
- ✅ React Router for client-side routing
- ✅ React Query for data fetching
- ✅ Symfony backend API ready
- ✅ Build pipeline configured
- ✅ Path aliases set up (`@/` imports)

**Next Steps:**
1. Receive home page screenshot
2. Create page components in `assets/controllers/react/front/pages/`
3. Set up routing in `App.js`
4. Build layouts and shared components
5. Style with TailwindCSS v4
6. Add shadcn components as needed

---

**Scan Complete! Ready for screenshot #1 - Home Page** 🚀
