#!/bin/bash
# Authentication System Quick Start Script
# This script helps you set up the authentication system quickly

echo "🔐 Authentication System Setup"
echo "=============================="
echo ""

# Colors for output
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
RED='\033[0;31m'
NC='\033[0m' # No Color

# Step 1: Install composer dependencies
echo -e "${YELLOW}Step 1: Installing Composer dependencies...${NC}"
composer require symfonycasts/verify-email-bundle
if [ $? -eq 0 ]; then
    echo -e "${GREEN}✓ Composer dependencies installed${NC}"
else
    echo -e "${RED}✗ Failed to install dependencies${NC}"
    exit 1
fi
echo ""

# Step 2: Check if .env.local exists
if [ ! -f .env.local ]; then
    echo -e "${YELLOW}Step 2: Creating .env.local...${NC}"
    
    # Generate a random secret
    SECRET=$(php -r "echo bin2hex(random_bytes(32));")
    
    cat > .env.local << EOL
###> Authentication System Configuration ###

# Database (update with your credentials)
DATABASE_URL="mysql://user:password@127.0.0.1:3306/dbname?serverVersion=8.0.32&charset=utf8mb4"

# Mailer (for development)
MAILER_DSN=smtp://localhost:1025

# Application URL
APP_URL=http://localhost:8000

# Application Secret (required for remember_me)
APP_SECRET=${SECRET}

###< Authentication System Configuration ###
EOL
    
    echo -e "${GREEN}✓ .env.local created with random APP_SECRET${NC}"
    echo -e "${YELLOW}⚠ Please update DATABASE_URL in .env.local with your database credentials${NC}"
else
    echo -e "${YELLOW}Step 2: .env.local already exists, skipping...${NC}"
    echo -e "${YELLOW}⚠ Make sure these variables are set:${NC}"
    echo "   - APP_SECRET"
    echo "   - APP_URL"
    echo "   - MAILER_DSN"
    echo "   - DATABASE_URL"
fi
echo ""

# Step 3: Run migrations
echo -e "${YELLOW}Step 3: Running database migrations...${NC}"
read -p "Run migrations now? (y/n) " -n 1 -r
echo
if [[ $REPLY =~ ^[Yy]$ ]]; then
    php bin/console doctrine:migrations:migrate --no-interaction
    if [ $? -eq 0 ]; then
        echo -e "${GREEN}✓ Migrations applied successfully${NC}"
    else
        echo -e "${RED}✗ Migration failed - check your database configuration${NC}"
        exit 1
    fi
else
    echo -e "${YELLOW}⚠ Skipping migrations - run manually: php bin/console doctrine:migrations:migrate${NC}"
fi
echo ""

# Step 4: Install npm dependencies
echo -e "${YELLOW}Step 4: Installing npm dependencies...${NC}"
if [ -f "package.json" ]; then
    npm install
    if [ $? -eq 0 ]; then
        echo -e "${GREEN}✓ npm dependencies installed${NC}"
    else
        echo -e "${RED}✗ npm install failed${NC}"
        exit 1
    fi
else
    echo -e "${YELLOW}⚠ package.json not found, skipping npm install${NC}"
fi
echo ""

# Step 5: Build assets
echo -e "${YELLOW}Step 5: Building frontend assets...${NC}"
if [ -f "package.json" ]; then
    npm run dev
    if [ $? -eq 0 ]; then
        echo -e "${GREEN}✓ Assets built successfully${NC}"
    else
        echo -e "${RED}✗ Asset build failed${NC}"
        exit 1
    fi
else
    echo -e "${YELLOW}⚠ package.json not found, skipping asset build${NC}"
fi
echo ""

# Step 6: Clear cache
echo -e "${YELLOW}Step 6: Clearing cache...${NC}"
php bin/console cache:clear
if [ $? -eq 0 ]; then
    echo -e "${GREEN}✓ Cache cleared${NC}"
else
    echo -e "${RED}✗ Cache clear failed${NC}"
fi
echo ""

# Final instructions
echo -e "${GREEN}================================${NC}"
echo -e "${GREEN}✓ Setup Complete!${NC}"
echo -e "${GREEN}================================${NC}"
echo ""
echo -e "${YELLOW}Next steps:${NC}"
echo ""
echo "1. Start Mailpit (for email testing):"
echo "   docker run -d -p 1025:1025 -p 8025:8025 axllent/mailpit"
echo "   View emails at: http://localhost:8025"
echo ""
echo "2. Start Symfony server:"
echo "   symfony server:start"
echo "   or: php -S localhost:8000 -t public/"
echo ""
echo "3. Test the authentication system:"
echo "   - Registration: http://localhost:8000/register"
echo "   - Login: http://localhost:8000/login"
echo "   - Password Reset: http://localhost:8000/reset-password"
echo ""
echo "4. Run tests:"
echo "   php bin/phpunit tests/Controller/AuthenticationFlowTest.php"
echo ""
echo -e "${YELLOW}📚 Documentation:${NC}"
echo "   - Quick Start: SETUP_AUTH.md"
echo "   - Reference: AUTHENTICATION.md"
echo "   - Complete Guide: docs/auth.md"
echo ""
echo -e "${GREEN}Happy coding! 🚀${NC}"
