# Authentication System Documentation

## Overview

This Symfony 7.3 application includes a complete, production-ready authentication system with the following features:

- **User Registration** with email confirmation
- **Login/Logout** with remember me functionality
- **Password Reset** via email with secure token-based flow
- **Email Verification** with expiring tokens (48 hours)
- **CSRF Protection** on all forms
- **Secure Password Hashing** using Symfony's password hasher
- **Modern UI** using Tailwind CSS and shadcn components

## Architecture

### Entities

#### User Entity (`src/Entity/User.php`)
- **Fields:**
  - `id`: Primary key
  - `email`: Unique email address (used as username)
  - `roles`: Array of roles (defaults to `ROLE_USER`)
  - `password`: Hashed password
  - `isVerified`: Boolean flag for email verification status
  - `createdAt`: Account creation timestamp

#### ResetPasswordToken Entity (`src/Entity/ResetPasswordToken.php`)
- **Fields:**
  - `id`: Primary key
  - `user`: Foreign key to User entity
  - `selector`: Public token selector (20 bytes, hex-encoded)
  - `hashedToken`: SHA-256 hash of verifier (40 bytes, hex-encoded)
  - `requestedAt`: Token creation timestamp
  - `expiresAt`: Token expiration timestamp (1 hour from creation)

### Services

#### EmailVerifier (`src/Service/EmailVerifier.php`)
Handles email confirmation using the SymfonyCasts VerifyEmailBundle:
- Generates signed URLs with expiration
- Validates email confirmation links
- Marks users as verified

#### ResetPasswordHelper (`src/Service/ResetPasswordHelper.php`)
Manages password reset tokens:
- Creates cryptographically secure tokens
- Validates tokens and checks expiration
- Removes used/expired tokens
- Uses split-token design (selector:verifier) for security

### Controllers

#### SecurityController (`src/Controller/SecurityController.php`)
- **Route:** `/login` - Display login form and handle authentication
- **Route:** `/logout` - Handle user logout

#### RegistrationController (`src/Controller/RegistrationController.php`)
- **Route:** `/register` - User registration form
- **Route:** `/verify/email` - Email verification callback

#### ResetPasswordController (`src/Controller/ResetPasswordController.php`)
- **Route:** `/reset-password` - Request password reset link
- **Route:** `/reset-password/reset/{token}` - Set new password

### Form Types

- **RegistrationFormType** - Email, password (min 8 chars), confirm password, terms checkbox
- **ResetPasswordRequestFormType** - Email field
- **ResetPasswordFormType** - New password and confirmation

### Templates

All templates use Tailwind CSS and shadcn design system for consistent, modern UI:

- `templates/_components/` - Reusable UI components
  - `auth_layout.html.twig` - Base layout for auth pages
  - `form_input.html.twig` - Styled input fields
  - `form_checkbox.html.twig` - Styled checkboxes
  - `button.html.twig` - Button variants
  - `alert.html.twig` - Flash message alerts

- `templates/security/login.html.twig` - Login form
- `templates/registration/register.html.twig` - Registration form
- `templates/registration/confirmation_email.html.twig` - Email verification email
- `templates/reset_password/request.html.twig` - Password reset request form
- `templates/reset_password/reset.html.twig` - Set new password form
- `templates/reset_password/email.html.twig` - Password reset email

## Installation & Setup

### 1. Install Required Packages

```bash
composer require symfonycasts/verify-email-bundle
```

### 2. Configure Environment Variables

Add these to your `.env.local` file:

```env
# Database Configuration
DATABASE_URL="mysql://user:password@127.0.0.1:3306/your_db?serverVersion=8.0.32&charset=utf8mb4"

# Mailer Configuration
# For development with Mailpit/Mailhog:
MAILER_DSN=smtp://localhost:1025

# For production (example with Gmail):
# MAILER_DSN=gmail+smtp://username:password@default

# For testing (emails not sent):
# MAILER_DSN=null://null

# Application URL (used in email links)
APP_URL=http://localhost:8000

# Ensure APP_SECRET is set (required for remember_me)
APP_SECRET=your-secret-key-here
```

### 3. Run Database Migrations

```bash
# Apply the authentication migration
php bin/console doctrine:migrations:migrate

# Or generate a fresh migration if you modified entities
php bin/console doctrine:migrations:diff
php bin/console doctrine:migrations:migrate
```

### 4. Build Frontend Assets

```bash
# Install dependencies
npm install

# Build assets for development
npm run dev

# Or watch for changes
npm run watch
```

### 5. Clear Cache

```bash
php bin/console cache:clear
```

## Usage

### User Registration Flow

1. User visits `/register`
2. Fills out email, password, and agrees to terms
3. System creates user account with `isVerified = false`
4. Verification email sent with signed URL (expires in 48 hours)
5. User clicks link in email
6. System verifies signature and marks user as verified
7. User can now log in

### Login Flow

1. User visits `/login`
2. Enters email and password
3. Optional: Check "Remember me" for persistent session (1 week)
4. System authenticates and redirects to home page
5. Access to `/user` routes now available

### Password Reset Flow

1. User visits `/reset-password`
2. Enters email address
3. System generates secure token (selector:verifier pair)
4. Reset email sent with link containing token (expires in 1 hour)
5. User clicks link to `/reset-password/reset/{token}`
6. Enters new password
7. System validates token, updates password, removes token
8. User can log in with new password

### Email Configuration

**For Development:**

Use Mailpit (recommended) or Mailhog to catch emails locally:

```bash
# Using Docker
docker run -d -p 1025:1025 -p 8025:8025 axllent/mailpit

# Configure .env.local
MAILER_DSN=smtp://localhost:1025

# View emails at http://localhost:8025
```

**For Production:**

Configure a real SMTP server or service:

```env
# Gmail (requires app password)
MAILER_DSN=gmail+smtp://username:app-password@default

# SendGrid
MAILER_DSN=sendgrid+smtp://apikey:YOUR_SENDGRID_API_KEY@default

# Amazon SES
MAILER_DSN=ses+smtp://ACCESS_KEY:SECRET_KEY@default?region=us-east-1
```

Update the "from" address in controllers:
- `RegistrationController.php` line 52
- `ResetPasswordController.php` line 62

## Security Features

### Password Hashing
- Uses Symfony's auto password hasher (bcrypt/argon2)
- Automatic rehashing on login if algorithm improves
- Passwords never stored in plain text

### CSRF Protection
- All forms include CSRF tokens
- Login form uses `authenticate` token name
- Symfony forms automatically handle CSRF

### Token Security
- **Email verification:** Signed URLs with HMAC, expires in 48 hours
- **Password reset:** Split-token design (selector:verifier), SHA-256 hashing, expires in 1 hour
- Tokens invalidated after use
- Old tokens automatically cleaned up

### Remember Me
- Configured in `security.yaml`
- 1 week lifetime (604800 seconds)
- Uses secret from `APP_SECRET` environment variable

### Access Control

Routes are protected in `config/packages/security.yaml`:

```yaml
access_control:
    - { path: ^/admin, roles: ROLE_ADMIN }
    - { path: ^/user, roles: ROLE_USER }
```

## Testing

### Run Tests

```bash
# Run all tests
php bin/phpunit

# Run authentication tests only
php bin/phpunit tests/Controller/AuthenticationFlowTest.php
```

### Manual Testing Checklist

#### Registration & Email Verification
- [ ] Visit `/register`
- [ ] Fill out registration form
- [ ] Submit and verify redirect to `/login`
- [ ] Check database: user created with `is_verified = 0`
- [ ] Open Mailpit (http://localhost:8025)
- [ ] Click verification link in email
- [ ] Verify user `is_verified` updated to `1`

#### Login
- [ ] Visit `/login`
- [ ] Enter valid credentials
- [ ] Check "Remember me"
- [ ] Submit and verify redirect to home
- [ ] Close browser and reopen - should still be logged in
- [ ] Clear cookies and verify logout

#### Password Reset
- [ ] Visit `/reset-password`
- [ ] Enter registered email
- [ ] Check Mailpit for reset email
- [ ] Click reset link
- [ ] Enter new password
- [ ] Verify password changed by logging in
- [ ] Verify old token cannot be reused

#### Access Control
- [ ] Logout
- [ ] Try to access `/user` route
- [ ] Verify redirect to `/login`
- [ ] Login and verify access granted

#### Error Handling
- [ ] Try registering with existing email
- [ ] Try login with wrong password
- [ ] Try password reset with non-existent email
- [ ] Try expired verification link
- [ ] Try expired reset token

## Customization

### Change Token Expiration Times

**Email verification (default 48 hours):**
Edit `config/packages/symfonycasts_verify_email.yaml`:
```yaml
symfonycasts_verify_email:
    lifetime: 172800 # seconds
```

**Password reset (default 1 hour):**
Edit `src/Service/ResetPasswordHelper.php`:
```php
private const TOKEN_LIFETIME = 3600; // seconds
```

### Change Remember Me Duration

Edit `config/packages/security.yaml`:
```yaml
remember_me:
    lifetime: 604800 # 1 week in seconds
```

### Customize Email Templates

Edit templates in `templates/registration/` and `templates/reset_password/`:
- `confirmation_email.html.twig`
- `email.html.twig`

### Customize Form Validation

Edit form types in `src/Form/`:
- `RegistrationFormType.php` - Change password min length, add custom validators
- `ResetPasswordFormType.php` - Add password complexity rules

### Customize UI Components

Edit Twig components in `templates/_components/`:
- `form_input.html.twig` - Modify input styling
- `button.html.twig` - Add new button variants
- `alert.html.twig` - Customize alert styles

## Commands

### Useful Symfony Commands

```bash
# Generate new migration after entity changes
php bin/console doctrine:migrations:diff

# Apply migrations
php bin/console doctrine:migrations:migrate

# Check database schema status
php bin/console doctrine:schema:validate

# Clear cache
php bin/console cache:clear

# List all routes
php bin/console debug:router

# View user in database
php bin/console doctrine:query:sql "SELECT * FROM user"

# Create admin user programmatically
php bin/console app:create-admin
```

### Create Admin User (Example)

Create `src/Command/CreateAdminCommand.php`:

```php
<?php

namespace App\Command;

use App\Entity\User;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Question\Question;
use Symfony\Component\PasswordHasher\Hasher\UserPasswordHasherInterface;

#[AsCommand(name: 'app:create-admin')]
class CreateAdminCommand extends Command
{
    public function __construct(
        private EntityManagerInterface $em,
        private UserPasswordHasherInterface $passwordHasher
    ) {
        parent::__construct();
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $helper = $this->getHelper('question');
        
        $email = $helper->ask($input, $output, new Question('Email: '));
        $password = $helper->ask($input, $output, new Question('Password: '));
        
        $user = new User();
        $user->setEmail($email);
        $user->setPassword($this->passwordHasher->hashPassword($user, $password));
        $user->setRoles(['ROLE_ADMIN']);
        $user->setIsVerified(true);
        
        $this->em->persist($user);
        $this->em->flush();
        
        $output->writeln('<info>Admin user created!</info>');
        
        return Command::SUCCESS;
    }
}
```

## Troubleshooting

### Emails Not Sending
- Check `MAILER_DSN` in `.env.local`
- Verify Mailpit/Mailhog is running
- Check logs: `var/log/dev.log`

### Migration Errors
- Run `php bin/console doctrine:schema:validate`
- Check database connection in `.env.local`
- Ensure database exists

### CSRF Token Invalid
- Clear cache: `php bin/console cache:clear`
- Check form has `{{ form_start() }}` and `{{ form_end() }}`
- Verify session is working

### Remember Me Not Working
- Ensure `APP_SECRET` is set in `.env`
- Clear browser cookies
- Check `security.yaml` configuration

### Access Denied After Login
- Check user has correct roles
- Verify `access_control` in `security.yaml`
- Check if user is verified (if required)

## File Structure

```
symfony-starter/
├── config/
│   └── packages/
│       ├── security.yaml                          # Security & firewall config
│       └── symfonycasts_verify_email.yaml        # Email verification config
├── migrations/
│   └── Version20250117000000.php                  # Auth system migration
├── src/
│   ├── Controller/
│   │   ├── RegistrationController.php
│   │   ├── ResetPasswordController.php
│   │   └── SecurityController.php
│   ├── Entity/
│   │   ├── ResetPasswordToken.php
│   │   └── User.php
│   ├── Form/
│   │   ├── RegistrationFormType.php
│   │   ├── ResetPasswordFormType.php
│   │   └── ResetPasswordRequestFormType.php
│   ├── Repository/
│   │   ├── ResetPasswordTokenRepository.php
│   │   └── UserRepository.php
│   └── Service/
│       ├── EmailVerifier.php
│       └── ResetPasswordHelper.php
├── templates/
│   ├── _components/
│   │   ├── alert.html.twig
│   │   ├── auth_layout.html.twig
│   │   ├── button.html.twig
│   │   ├── form_checkbox.html.twig
│   │   └── form_input.html.twig
│   ├── registration/
│   │   ├── confirmation_email.html.twig
│   │   └── register.html.twig
│   ├── reset_password/
│   │   ├── email.html.twig
│   │   ├── request.html.twig
│   │   └── reset.html.twig
│   └── security/
│       └── login.html.twig
└── tests/
    └── Controller/
        └── AuthenticationFlowTest.php
```

## Additional Resources

- [Symfony Security Documentation](https://symfony.com/doc/current/security.html)
- [VerifyEmailBundle Documentation](https://github.com/SymfonyCasts/verify-email-bundle)
- [Symfony Mailer Documentation](https://symfony.com/doc/current/mailer.html)
- [Doctrine Migrations](https://www.doctrine-project.org/projects/doctrine-migrations-bundle/en/latest/index.html)
