<?php

namespace App\Controller\Admin;

use App\Entity\User;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;

#[Route('/api/admin/settings', name: 'api_admin_settings_')]
class SettingsController extends AbstractController
{
    public function __construct(
        private readonly EntityManagerInterface $entityManager
    ) {
    }

    #[Route('/smtp', name: 'smtp_get', methods: ['GET'])]
    public function getSmtpSettings(): JsonResponse
    {
        $this->denyAccessUnlessGranted('ROLE_ADMIN');

        /** @var User $user */
        $user = $this->getUser();
        
        if (!$user instanceof User) {
            return $this->json([
                'success' => false,
                'message' => 'User not found'
            ], Response::HTTP_UNAUTHORIZED);
        }

        $config = $user->getSmtpConfig();
        
        // Don't send password in response, replace with placeholder if exists
        if (!empty($config['password'])) {
            $config['password'] = '********';
        }

        return $this->json([
            'success' => true,
            'data' => $config
        ]);
    }

    #[Route('/smtp', name: 'smtp_update', methods: ['POST'])]
    public function updateSmtpSettings(Request $request): JsonResponse
    {
        $this->denyAccessUnlessGranted('ROLE_ADMIN');

        /** @var User $user */
        $user = $this->getUser();
        
        if (!$user instanceof User) {
            return $this->json([
                'success' => false,
                'message' => 'User not found'
            ], Response::HTTP_UNAUTHORIZED);
        }

        $data = json_decode($request->getContent(), true);

        if (!$data) {
            return $this->json([
                'success' => false,
                'message' => 'Invalid JSON data'
            ], Response::HTTP_BAD_REQUEST);
        }

        // Validate required fields
        $errors = [];
        
        if (empty($data['host'])) {
            $errors['host'] = 'SMTP host is required';
        }
        
        if (empty($data['port']) || !is_numeric($data['port'])) {
            $errors['port'] = 'Valid SMTP port is required';
        }
        
        if (empty($data['username'])) {
            $errors['username'] = 'SMTP username is required';
        }
        
        // Only validate password if it's not the placeholder
        if (empty($data['password']) || $data['password'] === '********') {
            // Keep existing password
            $existingConfig = $user->getSmtpConfig();
            $data['password'] = $existingConfig['password'];
        }
        
        if (empty($data['from_email']) || !filter_var($data['from_email'], FILTER_VALIDATE_EMAIL)) {
            $errors['from_email'] = 'Valid from email is required';
        }

        if (!empty($errors)) {
            return $this->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $errors
            ], Response::HTTP_BAD_REQUEST);
        }

        try {
            $user->setSmtpConfig([
                'host' => $data['host'],
                'port' => (int) $data['port'],
                'encryption' => $data['encryption'] ?? 'tls',
                'username' => $data['username'],
                'password' => $data['password'],
                'from_email' => $data['from_email'],
                'from_name' => $data['from_name'] ?? 'Application',
            ]);

            $this->entityManager->persist($user);
            $this->entityManager->flush();

            return $this->json([
                'success' => true,
                'message' => 'SMTP settings updated successfully'
            ]);
        } catch (\Exception $e) {
            return $this->json([
                'success' => false,
                'message' => 'Failed to update settings: ' . $e->getMessage()
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    #[Route('/smtp/test', name: 'smtp_test', methods: ['POST'])]
    public function testSmtpConnection(Request $request): JsonResponse
    {
        $this->denyAccessUnlessGranted('ROLE_ADMIN');

        /** @var User $user */
        $user = $this->getUser();
        
        if (!$user instanceof User) {
            return $this->json([
                'success' => false,
                'message' => 'User not found'
            ], Response::HTTP_UNAUTHORIZED);
        }

        $data = json_decode($request->getContent(), true);

        if (!$data || empty($data['test_email'])) {
            return $this->json([
                'success' => false,
                'message' => 'Test email address is required'
            ], Response::HTTP_BAD_REQUEST);
        }

        try {
            // Here you would send a test email using the current SMTP settings
            // For now, we'll just validate the configuration exists
            $config = $user->getSmtpConfig();
            
            if (empty($config['host'])) {
                return $this->json([
                    'success' => false,
                    'message' => 'SMTP configuration not found. Please save your settings first.'
                ], Response::HTTP_BAD_REQUEST);
            }

            // TODO: Implement actual email sending test
            // You can inject MailerInterface and send a test email here

            return $this->json([
                'success' => true,
                'message' => 'SMTP configuration is valid. (Note: Actual email sending test not implemented yet)'
            ]);
        } catch (\Exception $e) {
            return $this->json([
                'success' => false,
                'message' => 'SMTP test failed: ' . $e->getMessage()
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }
}
