<?php

namespace App\Controller;

use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\KernelInterface;
use Symfony\Component\Routing\Annotation\Route;
use App\Service\Installer\SystemRequirementsChecker;
use App\Service\Installer\DatabaseConfigWriter;
use App\Service\Installer\SchemaInstaller;
use App\Service\Installer\AdminUserCreator;
use App\Service\Installer\SmtpConfigWriter;
use App\Service\Installer\AppConfigWriter;
use App\Service\Installer\InstallationStatusChecker;
use App\Service\Installer\EnvFileWriter;

#[Route('/install/api', name: 'installer_api_')]
class InstallerApiController extends AbstractController
{
    /**
     * Step 1: System Requirements Check
     */
    #[Route('/system-check', name: 'system_check', methods: ['GET'])]
    public function systemCheck(SystemRequirementsChecker $checker): JsonResponse
    {
        try {
            $result = $checker->check();
            return new JsonResponse(['success' => true, ...$result]);
        } catch (\Exception $e) {
            return new JsonResponse([
                'success' => false,
                'message' => 'Error checking system requirements: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Step 2: Save Database Configuration
     */
    #[Route('/database-config', name: 'database_config', methods: ['POST'])]
    public function databaseConfig(
        Request $request,
        DatabaseConfigWriter $writer,
        EnvFileWriter $envWriter
    ): JsonResponse {
        try {
            $data = json_decode($request->getContent(), true);

            if (!$data) {
                return new JsonResponse([
                    'success' => false,
                    'message' => 'Invalid JSON data'
                ], 400);
            }

            // Validate required fields (password can be empty for localhost)
            $required = ['db_name', 'host', 'port', 'db_user'];
            foreach ($required as $field) {
                if (!isset($data[$field]) || $data[$field] === '') {
                    return new JsonResponse([
                        'success' => false,
                        'message' => "Field {$field} is required"
                    ], 400);
                }
            }
            
            // Password must be present but can be empty
            if (!isset($data['password'])) {
                return new JsonResponse([
                    'success' => false,
                    'message' => 'Field password is required'
                ], 400);
            }

            // Test connection first
            $connectionTest = $writer->testConnection($data);
            if (!$connectionTest['success']) {
                return new JsonResponse($connectionTest, 400);
            }

            // Write configuration to config/db.yaml
            $writer->writeConfig($data);

            // Update .env file with DATABASE_URL
            if (!$envWriter->updateDatabaseUrl($data)) {
                return new JsonResponse([
                    'success' => false,
                    'message' => 'Database config saved but failed to update .env file. Please check file permissions.'
                ], 500);
            }

            return new JsonResponse([
                'success' => true,
                'message' => 'Database configuration saved successfully'
            ]);

        } catch (\Exception $e) {
            return new JsonResponse([
                'success' => false,
                'message' => 'Error saving database configuration: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Step 3: Install Database Tables
     */
    #[Route('/install-tables', name: 'install_tables', methods: ['POST'])]
    public function installTables(
        DatabaseConfigWriter $dbWriter,
        SchemaInstaller $installer
    ): JsonResponse {
        try {
            // Read database configuration
            $dbConfig = $dbWriter->readConfig();

            if (!$dbConfig || !$dbWriter->validateConfig($dbConfig)) {
                return new JsonResponse([
                    'success' => false,
                    'message' => 'Database configuration not found. Complete Step 2 first.'
                ], 400);
            }

            // Install schema
            $result = $installer->install($dbConfig);

            return new JsonResponse($result, $result['success'] ? 200 : 500);

        } catch (\Exception $e) {
            return new JsonResponse([
                'success' => false,
                'message' => 'Error installing tables: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Step 4: Create Admin User
     */
    #[Route('/create-admin', name: 'create_admin', methods: ['POST'])]
    public function createAdmin(
        Request $request,
        DatabaseConfigWriter $dbWriter,
        AdminUserCreator $creator
    ): JsonResponse {
        try {
            $data = json_decode($request->getContent(), true);

            if (!$data) {
                return new JsonResponse([
                    'success' => false,
                    'message' => 'Invalid JSON data'
                ], 400);
            }

            // Read database configuration
            $dbConfig = $dbWriter->readConfig();

            if (!$dbConfig || !$dbWriter->validateConfig($dbConfig)) {
                return new JsonResponse([
                    'success' => false,
                    'message' => 'Database configuration not found. Complete Step 2 first.'
                ], 400);
            }

            // Create admin user
            $result = $creator->createAdmin($dbConfig, $data);

            return new JsonResponse($result, $result['success'] ? 200 : 500);

        } catch (\Exception $e) {
            return new JsonResponse([
                'success' => false,
                'message' => 'Error creating admin user: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Step 5: Save SMTP Configuration
     */
    #[Route('/smtp-config', name: 'smtp_config', methods: ['POST'])]
    public function smtpConfig(
        Request $request,
        DatabaseConfigWriter $dbWriter,
        SmtpConfigWriter $smtpWriter
    ): JsonResponse {
        try {
            $data = json_decode($request->getContent(), true);

            if (!$data) {
                return new JsonResponse([
                    'success' => false,
                    'message' => 'Invalid JSON data'
                ], 400);
            }

            // Read database configuration
            $dbConfig = $dbWriter->readConfig();

            if (!$dbConfig || !$dbWriter->validateConfig($dbConfig)) {
                return new JsonResponse([
                    'success' => false,
                    'message' => 'Database configuration not found. Complete Step 2 first.'
                ], 400);
            }

            // Save SMTP configuration
            $result = $smtpWriter->saveSmtpConfig($dbConfig, $data);

            return new JsonResponse($result, $result['success'] ? 200 : 500);

        } catch (\Exception $e) {
            return new JsonResponse([
                'success' => false,
                'message' => 'Error saving SMTP configuration: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Step 6: Save Application Configuration
     */
    #[Route('/app-config', name: 'app_config', methods: ['POST'])]
    public function appConfig(
        Request $request,
        AppConfigWriter $writer
    ): JsonResponse {
        try {
            $data = json_decode($request->getContent(), true);

            if (!$data) {
                return new JsonResponse([
                    'success' => false,
                    'message' => 'Invalid JSON data'
                ], 400);
            }

            // Write configuration
            $result = $writer->writeConfig($data);

            // If result has success=false, return it with proper status
            if (!$result['success']) {
                return new JsonResponse($result, 500);
            }

            return new JsonResponse($result, 200);

        } catch (\Throwable $e) {
            return new JsonResponse([
                'success' => false,
                'message' => 'Error saving app configuration: ' . $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ], 500);
        }
    }

    /**
     * Step 7: Clear Cache (Final Step)
     * Note: In production mode, this may fail because the app is running from the cache.
     * Users should manually clear cache after installation if this fails.
     */
    // #[Route('/clear-cache', name: 'clear_cache', methods: ['POST'])]
    // public function clearCache(KernelInterface $kernel): JsonResponse
    // {
    //     try {
    //         $filesystem = new \Symfony\Component\Filesystem\Filesystem();
    //         $projectDir = $kernel->getProjectDir();
    //         $prodCacheDir = $projectDir . '/var/cache/prod';
            
    //         // Check if we're running in prod mode
    //         $env = $kernel->getEnvironment();
            
    //         if ($env === 'prod') {
    //             // In prod mode, we can't delete the cache we're running from
    //             // Return a friendly message instead of failing
    //             return new JsonResponse([
    //                 'success' => false,
    //                 'message' => 'Cannot clear cache while running in production mode. Please manually clear the cache by deleting the /var/cache/prod folder or running: php bin/console cache:clear'
    //             ], 200); // Return 200 so it's not treated as a server error
    //         }
            
    //         if (is_dir($prodCacheDir)) {
    //             // Delete the entire prod cache directory (only works in dev mode)
    //             $filesystem->remove($prodCacheDir);
                
    //             return new JsonResponse([
    //                 'success' => true,
    //                 'message' => 'Cache cleared successfully'
    //             ]);
    //         }
            
    //         return new JsonResponse([
    //             'success' => true,
    //             'message' => 'No cache to clear'
    //         ]);
            
    //     } catch (\Exception $e) {
    //         return new JsonResponse([
    //             'success' => false,
    //             'message' => 'Error clearing cache: ' . $e->getMessage() . '. Please manually clear the cache by deleting the /var/cache/prod folder.'
    //         ], 200); // Return 200 so it's not treated as a server error
    //     }
    // }

    /**
     * Get Installation Status
     */
    #[Route('/status', name: 'status', methods: ['GET'])]
    public function status(InstallationStatusChecker $statusChecker): JsonResponse
    {
        try {
            $result = $statusChecker->getStatus();
            return new JsonResponse(['success' => true, ...$result]);
        } catch (\Exception $e) {
            return new JsonResponse([
                'success' => false,
                'message' => 'Error checking installation status: ' . $e->getMessage()
            ], 500);
        }
    }
}
