<?php

namespace App\Repository;

use App\Entity\Settings;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\Persistence\ManagerRegistry;

/**
 * @extends ServiceEntityRepository<Settings>
 */
class SettingsRepository extends ServiceEntityRepository
{
    public function __construct(ManagerRegistry $registry)
    {
        parent::__construct($registry, Settings::class);
    }

    public function findByKey(string $key): ?Settings
    {
        return $this->findOneBy(['settingKey' => $key]);
    }

    public function getValue(string $key, ?string $default = null): ?string
    {
        $setting = $this->findByKey($key);
        
        return $setting ? $setting->getSettingValue() : $default;
    }

    public function setValue(string $key, ?string $value): void
    {
        $setting = $this->findByKey($key);
        
        if (!$setting) {
            $setting = new Settings();
            $setting->setSettingKey($key);
        }
        
        $setting->setSettingValue($value);
        $setting->setUpdatedAt(new \DateTimeImmutable());
        
        $this->getEntityManager()->persist($setting);
        $this->getEntityManager()->flush();
    }

    public function getSmtpConfig(): array
    {
        return [
            'host' => $this->getValue('smtp_host', ''),
            'port' => (int) $this->getValue('smtp_port', '587'),
            'encryption' => $this->getValue('smtp_encryption', 'tls'),
            'username' => $this->getValue('smtp_username', ''),
            'password' => $this->getValue('smtp_password', ''),
            'from_email' => $this->getValue('smtp_from_email', 'noreply@example.com'),
            'from_name' => $this->getValue('smtp_from_name', 'Application'),
        ];
    }

    public function setSmtpConfig(array $config): void
    {
        $this->setValue('smtp_host', $config['host'] ?? '');
        $this->setValue('smtp_port', (string) ($config['port'] ?? 587));
        $this->setValue('smtp_encryption', $config['encryption'] ?? 'tls');
        $this->setValue('smtp_username', $config['username'] ?? '');
        $this->setValue('smtp_password', $config['password'] ?? '');
        $this->setValue('smtp_from_email', $config['from_email'] ?? 'noreply@example.com');
        $this->setValue('smtp_from_name', $config['from_name'] ?? 'Application');
    }
}
