<?php

namespace App\Service;

use App\Entity\User;
use Symfony\Bundle\SecurityBundle\Security;
use Symfony\Component\Mailer\Envelope;
use Symfony\Component\Mailer\MailerInterface;
use Symfony\Component\Mailer\Transport;
use Symfony\Component\Mailer\Transport\TransportInterface;
use Symfony\Component\Mime\Email;
use Symfony\Component\Mime\RawMessage;

class DynamicMailer implements MailerInterface
{
    private ?TransportInterface $transport = null;
    private array $smtpConfig = [];

    public function __construct(
        private readonly Security $security,
        private readonly MailerInterface $defaultMailer
    ) {
    }

    private function getTransport(): TransportInterface
    {
        $config = $this->getSmtpConfig();
        
        // If no SMTP host configured, use default mailer
        if (empty($config['host'])) {
            return $this->defaultMailer;
        }

        // Check if config changed
        if ($this->transport === null || $config !== $this->smtpConfig) {
            $this->smtpConfig = $config;
            
            // Build DSN from database config
            $dsn = $this->buildDsn($config);
            $this->transport = Transport::fromDsn($dsn);
        }

        return $this->transport;
    }

    private function buildDsn(array $config): string
    {
        $encryption = $config['encryption'] ?: 'tls';
        $scheme = 'smtp';
        
        if ($encryption === 'ssl') {
            $scheme = 'smtps';
        }

        $dsn = sprintf(
            '%s://%s:%s@%s:%d',
            $scheme,
            urlencode($config['username']),
            urlencode($config['password']),
            $config['host'],
            $config['port']
        );

        // Add encryption option for TLS
        if ($encryption === 'tls') {
            $dsn .= '?encryption=tls';
        }

        return $dsn;
    }

    public function send(RawMessage $message, ?Envelope $envelope = null): void
    {
        // Set from address if it's an Email object and from is not already set
        if ($message instanceof Email && count($message->getFrom()) === 0) {
            $config = $this->getSmtpConfig();
            $message->from($config['from_email']);
            
            if (!empty($config['from_name'])) {
                $from = $message->getFrom()[0];
                $message->from($from->withName($config['from_name']));
            }
        }

        $this->getTransport()->send($message, $envelope);
    }

    public function getSmtpConfig(): array
    {
        $user = $this->security->getUser();
        
        if (!$user instanceof User) {
            // No user logged in or not a User instance, return empty config
            // This will cause fallback to default mailer
            return [
                'host' => '',
                'port' => 587,
                'encryption' => 'tls',
                'username' => '',
                'password' => '',
                'from_email' => '',
                'from_name' => 'Application',
            ];
        }

        return $user->getSmtpConfig();
    }
}
