<?php

namespace App\Service\Installer;

use Symfony\Component\Filesystem\Filesystem;
use Symfony\Component\Yaml\Yaml;

class AppConfigWriter
{
    private array $appConfig;
    private string $installMarkerPath;
    private string $projectDir;

    public function __construct(array $appConfig, string $installMarkerPath, string $projectDir)
    {
        $this->appConfig = $appConfig;
        $this->installMarkerPath = $installMarkerPath;
        $this->projectDir = $projectDir;
    }

    public function writeConfig(array $configData): array
    {
        try {
            $configPath = $this->appConfig['config_path'];
            
            // Ensure directory exists
            $dir = dirname($configPath);
            if (!is_dir($dir)) {
                if (!mkdir($dir, 0755, true)) {
                    throw new \Exception("Failed to create config directory: {$dir}");
                }
            }

            // Build configuration
            $config = ['parameters' => []];

            // Standard parameters
            if (isset($configData['base_url'])) {
                $config['parameters']['app.base_url'] = rtrim($configData['base_url'], '/');
            }

            // Always set base_path to empty since base_url contains the full path
            // This prevents path duplication in asset() calls
            $config['parameters']['app.base_path'] = '';

            // Build assets base URL
            // If base_url already contains the path (e.g., https://domain.com/subfolder),
            // don't add base_path again
            if (isset($configData['base_url'])) {
                $baseUrl = rtrim($configData['base_url'], '/');
                $config['parameters']['app.assets_base_url'] = $baseUrl . '/public';
            }

            // Additional custom parameters from bundle configuration
            foreach ($this->appConfig['parameters'] as $param) {
                $paramName = $param['name'];
                if (isset($configData[$paramName])) {
                    $config['parameters'][$paramName] = $configData[$paramName];
                } elseif (isset($param['default'])) {
                    $config['parameters'][$paramName] = $param['default'];
                }
            }

            // Write YAML file
            $yamlContent = Yaml::dump($config, 4, 2);
            if (file_put_contents($configPath, $yamlContent) === false) {
                throw new \Exception("Failed to write config file: {$configPath}");
            }

            // Update services.yaml to import app_config.yaml if needed
            $this->updateServicesYaml();

            // Note: Cache clearing is handled by the installer via /install/api/clear-cache
            // after this step completes to avoid 500 errors during config write

            // Create installation marker
            $this->createInstallMarker();

            return [
                'success' => true,
                'message' => 'Application configuration saved successfully',
                'config' => $config['parameters']
            ];

        } catch (\Throwable $e) {
            return [
                'success' => false,
                'message' => 'Error saving configuration: ' . $e->getMessage() . ' (Line: ' . $e->getLine() . ')'
            ];
        }
    }

    private function updateServicesYaml(): void
    {
        $servicesPath = dirname($this->appConfig['config_path']) . '/services.yaml';
        
        if (!file_exists($servicesPath)) {
            return;
        }

        $servicesContent = file_get_contents($servicesPath);
        
        // Check if import already exists
        if (strpos($servicesContent, 'app_config.yaml') !== false) {
            return;
        }

        // Add import at the beginning
        $importLine = "# Import application configuration\nimports:\n    - { resource: app_config.yaml }\n\n";
        $servicesContent = $importLine . $servicesContent;
        
        file_put_contents($servicesPath, $servicesContent);
    }

    private function createInstallMarker(): void
    {
        $markerPath = $this->installMarkerPath;
        $dir = dirname($markerPath);
        
        if (!is_dir($dir)) {
            mkdir($dir, 0755, true);
        }

        $content = sprintf(
            "Installation completed: %s\nInstaller Bundle: Webberdoo\\InstallerBundle\n",
            date('Y-m-d H:i:s')
        );
        
        file_put_contents($markerPath, $content);
    }

    public function isInstalled(): bool
    {
        return file_exists($this->installMarkerPath);
    }

    private function clearCache(): void
    {
        $filesystem = new Filesystem();
        
        // Clear only the prod cache directory contents, not the directory itself
        $prodCacheDir = $this->projectDir . '/var/cache/prod';
        
        if (is_dir($prodCacheDir)) {
            try {
                // Get all items in the cache directory
                $items = glob($prodCacheDir . '/*');
                if ($items) {
                    foreach ($items as $item) {
                        $filesystem->remove($item);
                    }
                }
            } catch (\Exception $e) {
                // If cache clearing fails, it's not critical
                // The user can manually clear it with: php bin/console cache:clear
            }
        }
    }
}
