<?php

namespace App\Service;

use App\Entity\ResetPasswordToken;
use App\Entity\User;
use App\Repository\ResetPasswordTokenRepository;
use Doctrine\ORM\EntityManagerInterface;

class ResetPasswordHelper
{
    private const TOKEN_LIFETIME = 3600; // 1 hour in seconds

    public function __construct(
        private readonly EntityManagerInterface $entityManager,
        private readonly ResetPasswordTokenRepository $tokenRepository
    ) {
    }

    public function createToken(User $user): string
    {
        // Remove any existing tokens for this user
        $this->tokenRepository->removeAllForUser($user);

        // Generate cryptographically secure random bytes
        $selector = bin2hex(random_bytes(20));
        $verifier = bin2hex(random_bytes(20));

        $token = new ResetPasswordToken();
        $token->setUser($user);
        $token->setSelector($selector);
        $token->setHashedToken(hash('sha256', $verifier));
        $token->setExpiresAt(
            (new \DateTimeImmutable())->modify(sprintf('+%d seconds', self::TOKEN_LIFETIME))
        );

        $this->entityManager->persist($token);
        $this->entityManager->flush();

        // Return the full token (selector:verifier) that will be sent in the email
        return $selector . ':' . $verifier;
    }

    public function validateToken(string $fullToken): ?ResetPasswordToken
    {
        $parts = explode(':', $fullToken);
        if (count($parts) !== 2) {
            return null;
        }

        [$selector, $verifier] = $parts;

        $token = $this->tokenRepository->findOneBySelector($selector);

        if (!$token || $token->isExpired()) {
            return null;
        }

        // Verify the token matches
        if (!hash_equals($token->getHashedToken(), hash('sha256', $verifier))) {
            return null;
        }

        return $token;
    }

    public function removeToken(ResetPasswordToken $token): void
    {
        $this->entityManager->remove($token);
        $this->entityManager->flush();
    }

    public function cleanExpiredTokens(): int
    {
        return $this->tokenRepository->removeExpiredTokens();
    }
}
