<?php

namespace App\Tests\Controller;

use App\Entity\User;
use App\Repository\UserRepository;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;

class AuthenticationFlowTest extends WebTestCase
{
    public function testRegistrationPage(): void
    {
        $client = static::createClient();
        $crawler = $client->request('GET', '/register');

        $this->assertResponseIsSuccessful();
        $this->assertSelectorTextContains('h2', 'Create your account');
        $this->assertSelectorExists('input[name="registration_form[email]"]');
        $this->assertSelectorExists('input[name="registration_form[plainPassword][first]"]');
        $this->assertSelectorExists('input[name="registration_form[plainPassword][second]"]');
        $this->assertSelectorExists('input[name="registration_form[agreeTerms]"]');
    }

    public function testSuccessfulRegistration(): void
    {
        $client = static::createClient();
        $crawler = $client->request('GET', '/register');

        $form = $crawler->selectButton('Create account')->form([
            'registration_form[email]' => 'test@example.com',
            'registration_form[plainPassword][first]' => 'TestPassword123!',
            'registration_form[plainPassword][second]' => 'TestPassword123!',
            'registration_form[agreeTerms]' => true,
        ]);

        $client->submit($form);

        $this->assertResponseRedirects('/login');
        
        // Verify user was created
        $userRepository = static::getContainer()->get(UserRepository::class);
        $user = $userRepository->findOneBy(['email' => 'test@example.com']);
        
        $this->assertNotNull($user);
        $this->assertFalse($user->isVerified(), 'User should not be verified initially');
        $this->assertInstanceOf(\DateTimeImmutable::class, $user->getCreatedAt());
    }

    public function testLoginPage(): void
    {
        $client = static::createClient();
        $crawler = $client->request('GET', '/login');

        $this->assertResponseIsSuccessful();
        $this->assertSelectorTextContains('h2', 'Sign in to your account');
        $this->assertSelectorExists('input[name="_username"]');
        $this->assertSelectorExists('input[name="_password"]');
        $this->assertSelectorExists('input[name="_csrf_token"]');
    }

    public function testLoginWithValidCredentials(): void
    {
        $client = static::createClient();
        
        // Create a test user
        $userRepository = static::getContainer()->get(UserRepository::class);
        $user = new User();
        $user->setEmail('login@example.com');
        $user->setIsVerified(true);
        
        $passwordHasher = static::getContainer()->get('security.user_password_hasher');
        $user->setPassword($passwordHasher->hashPassword($user, 'TestPassword123!'));
        
        $entityManager = static::getContainer()->get('doctrine')->getManager();
        $entityManager->persist($user);
        $entityManager->flush();

        // Attempt login
        $crawler = $client->request('GET', '/login');
        $form = $crawler->selectButton('Sign in')->form([
            '_username' => 'login@example.com',
            '_password' => 'TestPassword123!',
        ]);

        $client->submit($form);

        $this->assertResponseRedirects();
    }

    public function testLoginWithInvalidCredentials(): void
    {
        $client = static::createClient();
        $crawler = $client->request('GET', '/login');

        $form = $crawler->selectButton('Sign in')->form([
            '_username' => 'nonexistent@example.com',
            '_password' => 'WrongPassword',
        ]);

        $client->submit($form);

        $this->assertResponseRedirects('/login');
        $client->followRedirect();
        $this->assertSelectorExists('.alert'); // Error message should be shown
    }

    public function testPasswordResetRequestPage(): void
    {
        $client = static::createClient();
        $crawler = $client->request('GET', '/reset-password');

        $this->assertResponseIsSuccessful();
        $this->assertSelectorTextContains('h2', 'Reset your password');
        $this->assertSelectorExists('input[name="reset_password_request_form[email]"]');
    }

    public function testPasswordResetRequest(): void
    {
        $client = static::createClient();
        
        // Create a test user
        $userRepository = static::getContainer()->get(UserRepository::class);
        $user = new User();
        $user->setEmail('reset@example.com');
        $user->setIsVerified(true);
        
        $passwordHasher = static::getContainer()->get('security.user_password_hasher');
        $user->setPassword($passwordHasher->hashPassword($user, 'OldPassword123!'));
        
        $entityManager = static::getContainer()->get('doctrine')->getManager();
        $entityManager->persist($user);
        $entityManager->flush();

        // Request password reset
        $crawler = $client->request('GET', '/reset-password');
        $form = $crawler->selectButton('Send reset link')->form([
            'reset_password_request_form[email]' => 'reset@example.com',
        ]);

        $client->submit($form);

        $this->assertResponseRedirects('/login');
        
        // Follow redirect and check for success message
        $crawler = $client->followRedirect();
        $this->assertSelectorExists('.alert'); // Success message
    }

    public function testAccessControlForUserArea(): void
    {
        $client = static::createClient();
        $client->request('GET', '/user');

        // Should redirect to login if not authenticated
        $this->assertResponseRedirects('/login');
    }
}
